## ##########################################################
## General utility functions
## ##########################################################
                
In <- function(x){0 + x}

## ######################################

vmin <- function(x,y){ c(apply(cbind(x,y), 1, min)) }

## ######################################

vmax <- function(x,y){ c(apply(cbind(x,y), 1, max)) }

## ######################################

fn <- function(x){(1:length(x))[x == 1]}

## ######################################

toz <- function(x){x + (pi/2) + 2 * pi * (x <= (-pi/2))}

## ######################################

bearing <- function(lon, lat, lon.ini, lat.ini){

  ## http://www.movable-type.co.uk/scripts/latlong.html
  ## corrected 8 December 2013: in RADIANS, not degrees
  ##                         : missing cos(lat) term
  ##                         : checked order of atan2 arguments
  
  lon <- lon * (pi / 180)
  
  lat <- lat * (pi / 180)
  
  lon.ini <- lon.ini * (pi / 180)
  
  lat.ini <- lat.ini * (pi / 180)
  
  londiff <- lon - lon.ini
  
  a1 <- sin(londiff) * cos(lat)
  
  a2 <- cos(lat.ini) * sin(lat) - sin(lat.ini) * cos(lat) * cos(londiff)
  
  atan2(a1, a2)
}

## ######################################

geodist3 <- function(Eto, Nto, Efrom, Nfrom){
  
  ## Borrowed from the 'GMT' package on CRAN
  ## Rewritten 11 December 2013 to deal with warning messages coming from applying acos to numbers very slightly bigger than 1
  
  rad <- 180 / pi
  
  zot <- sin(Nfrom / rad)*sin(Nto / rad)+cos(Nfrom / rad)*cos(Nto / rad)*cos((Efrom - Eto) / rad)
  
  tmp <- rep(NA, length(zot))

  tmp[zot > 0.999999999] <- 0

  tmp[zot <= 0.999999999] <- acos(zot[zot <= 0.999999999])
  
  60 * rad * 1.852 * tmp
}

## ######################################

cogeb <- function(x,y){

  ## Match against the GEBCO grid
  
  x <- x[x >= -15 & x <= 10]

  y <- y[y >= 43 & y <= 69]
  
  xnew <- round(1 + (119.96 * (x + 15)))
  ynew <- round(1 + (119.9615 * (y - 43)))
  
  if(length(x) == 1){
    
    out <- c(xnew, ynew)
  }
  else{
    
    out <- cbind(xnew, ynew)
  }

  out
}

## ##########################################################
## Specific functions:
##  sigfun, vsdm, vsdm.summary, vsdm.runmodel, vsdm.sensitivity,
##  make.prey.map, simlocs, sim.batch.SPAs, sim.one.SPA,
##  barrierdist, add.avoid, otherbirds, sim.all.SPAs,
##  getsummary.zones, getlims, getzone, addWF, plotWF
## ##########################################################

sigfun <- function(x, a, mu, lambda){ a * exp(-exp((mu * exp(1) / a) * (lambda - x) + 1)) }

## ###############################################################################################

vsdm <- function(spatdat, pars, total.prey, min.prey, timestep=24, seedval.behaviour, puffin=FALSE){

  set.seed(seedval.behaviour)

  npairs <- dim(spatdat$Depth)[1]

  ntimes <- dim(spatdat$Depth)[3]
  
  ## ##################################################
  ## --> Set up

  adult.dee <- adult.mass <- absolute.prey.density <- forage.depth <- diving.depth <- array(dim=c(npairs, 2, ntimes))

  independent.intake.rate <- final.intake.rate <- energy.gained.per.hour <- array(dim=c(npairs, 2, ntimes))
  
  hours.for.requirement <- travel.hours.per.trip <- time.nest.minimum <- available.time <- array(dim=c(npairs, 2, ntimes))

  flying.hours <- hours.open.to.forage <- foraging.hours <- surplus.hours <- time.nest <- time.rest <- array(dim=c(npairs, 2, ntimes))

  total.energy.gained <- energy.gap <- energy.to.adult <- energy.to.chick <- energy.requirement <- array(dim=c(npairs, 2, ntimes))

  adult.mass <- adult.dee <- nforagetrips <- avoiddist <- array(dim=c(npairs, 2, ntimes))
  
  chick.mass <- abandoned <- total.energy.to.chick <- time.chick.unattended <- array(dim=c(npairs, ntimes))

  chick.mortality.from.unattendance <- chick.mortality.from.lowmass <- chick.mortality.from.abandonment <- array(dim=c(npairs, ntimes))
    
  superchick.mass <- rep(NA, ntimes)

  ## ##################################################
  ## --> Simulation

  chick.dee <- rep(pars$chick.DEE, npairs)

  ## ##################################################
  ## --> Prey

  spatdat$total.prey <- spatdat$min.prey <- rep(NA, npairs)

  for(l in 1:length(unique(spatdat$SPA))){
    
    spatdat$total.prey[spatdat$SPA == unique(spatdat$SPA)[l]] <- total.prey[l]

    spatdat$min.prey[spatdat$SPA == unique(spatdat$SPA)[l]] <- min.prey[l]
  }
    
  ## ##################################################
  ## --> First time point (adult.dee, abandoned, adult.mass, chick.mass)

  for(j in 1:2){
    
    adult.dee[,j,1] <- rnorm(npairs, pars$adult.DEE.mn, pars$adult.DEE.sd)
  
    adult.mass[,j,1] <- rnorm(npairs, pars$BM.adult.mn, pars$BM.adult.sd)
  }
  
  chick.mass[,1] <- rnorm(npairs, pars$BM.chick.mn, pars$BM.chick.sd)

  abandoned[,1] <- rep(0, npairs)
  
  superchick.mass[1] <- mean(chick.mass[,1])

  ## ##################################################
  ## --> Loop over time points
  ##
  for(t in 2:ntimes){

    print.noquote(paste(t, ".....", date()))
    
    ## ---> loop over adults
    ##
    for(j in 1:2){
      
    ##  print.noquote(paste(j,t)) ; browser()
      
      ## #################################
      ## (1) Energy intake rate
      ## #################################
      
      absolute.prey.density[,j,t] <- spatdat$total.prey * spatdat$Prey.density[,j,t]

      absolute.prey.density[,j,t] <- absolute.prey.density[,j,t] * (absolute.prey.density[,j,t] > spatdat$min.prey) + spatdat$min.prey * (absolute.prey.density[,j,t] <= spatdat$min.prey) 
      
      forage.depth[,j,t] <- rnorm(npairs, pars$forage.depth.mn, pars$forage.depth.sd)
      
      diving.depth[,j,t] <- spatdat$Depth[,j,t] * (1 - pars$pelagic) + forage.depth[,j,t] * pars$pelagic

      if(pars$diving.eff1 == 0 & pars$diving.eff2 == 0){

        diving.part <- 1
      }
      else{

        diving.part <- pars$diving.eff1 + pars$diving.eff2 * diving.depth[,j,t]
      }
      
      independent.intake.rate[,j,t] <- sigfun(absolute.prey.density[,j,t],
                                              pars$IR.max, pars$IR.mu, pars$IR.lambda) * diving.part 
      
      final.intake.rate[,j,t] <- independent.intake.rate[,j,t] * (spatdat$Birds.in.square[,j,t] ^ (-pars$m))
      
      energy.gained.per.hour[,j,t] <- final.intake.rate[,j,t] * pars$assim.eff * pars$energy.prey * 60 
      
      ## #################################
      ## (2) Energy requirement
      ## #################################
          
      energy.requirement[,j,t] <- adult.dee[,j,t-1] + (chick.dee / 2) * (1 - abandoned[,t-1]) # **************************
      
      ## #################################
      ## (3) Time budget
      ## #################################
      
      hours.for.requirement[,j,t] <- energy.requirement[,j,t] / energy.gained.per.hour[,j,t] 
      
      travel.hours.per.trip[,j,t] <- (2 * spatdat$Dist2Colony[,j,t] + spatdat$AvoidDist[,j,t] + spatdat$DispDist[,j,t]) / (pars$flight.msec * 3.6) 
             
      time.nest.minimum[,j,t] <- (timestep / 2) * (adult.mass[,j,t-1] > 0.9 * adult.mass[,j,1]) 
      
      available.time[,j,t] <- timestep - time.nest.minimum[,j,t] - pars$time.rest.minimum 
            
      test2 <- ((available.time[,j,t] - 2 * travel.hours.per.trip[,j,t]) > hours.for.requirement[,j,t])
      
      test3 <- ((available.time[,j,t] - 3 * travel.hours.per.trip[,j,t]) > hours.for.requirement[,j,t])

      test4 <- puffin & ((available.time[,j,t] - 4 * travel.hours.per.trip[,j,t]) > hours.for.requirement[,j,t]) 

      nextra <- (1 - abandoned[,t-1]) * (test2 + test3 + test4) 
      
      nforagetrips[,j,t] <- 1 + nextra 
      
      flying.hours[,j,t] <- travel.hours.per.trip[,j,t] * nforagetrips[,j,t]
      
      hours.open.to.forage[,j,t] <- available.time[,j,t] - flying.hours[,j,t]

      foraging.hours[,j,t] <- vmin(hours.open.to.forage[,j,t], hours.for.requirement[,j,t])

      surplus.hours[,j,t] <- hours.open.to.forage[,j,t] - foraging.hours[,j,t]

      time.nest[,j,t] <- time.nest.minimum[,j,t] + surplus.hours[,j,t] 

      time.rest[,j,t] <- pars$time.rest.minimum
                                                               
      ## #################################
      ## (4) Total intake rate
      ## #################################

      total.energy.gained[,j,t] <- energy.gained.per.hour[,j,t] * foraging.hours[,j,t]

      energy.gap[,j,t] <- energy.requirement[,j,t] - total.energy.gained[,j,t]

      energy.to.adult[,j,t] <- adult.dee[,j,t-1] - energy.gap[,j,t] * (1 - pars$adult.priority)^(1 - abandoned[,t-1]) 

      energy.to.chick[,j,t] <- (0.5 * chick.dee - energy.gap[,j,t] * pars$adult.priority) * (1 - abandoned[,t-1])

      shortfall <- -energy.to.chick[,j,t] * (energy.to.chick[,j,t] < 0) 
      
      energy.to.adult[,j,t] <- energy.to.adult[,j,t] - shortfall
      
      energy.to.chick[,j,t] <- energy.to.chick[,j,t] + shortfall 
      
      ## #################################
      ## (5) Adult outcomes
      ## #################################
           
      propn <- energy.to.adult[,j,t] / adult.dee[,j,t-1]

      adult.mass[,j,t] <-  adult.mass[,j,t-1]  + (adult.mass[,j,t-1]^(pars$adult.mass.a * propn)) - (adult.mass[,j,1] ^ pars$adult.mass.a)
      
      adult.dee[,j,t] <- (((pars$energy.flight * flying.hours[,j,t]) + (pars$energy.forage * foraging.hours[,j,t]) + (pars$energy.nest * time.nest[,j,t])
                           + (pars$energy.searest * time.rest[,j,t])) / timestep + (pars$energy.warming)) * (timestep / 24) 
    }
    ## <--- loop over adults
    ##
    
    ## #################################
    ## (6) Chick outcomes
    ## #################################

    total.energy.to.chick[,t] <- energy.to.chick[,1,t] + energy.to.chick[,2,t]

    time.chick.unattended[,t] <- vmax(timestep - time.nest[,1,t] - time.nest[,2,t], 0)
    
    chick.mass[,t] <- chick.mass[,t-1] + sigfun(total.energy.to.chick[,t], pars$chick.mass.a, pars$chick.mass.mu, pars$chick.mass.lambda)
    
    superchick.mass[t] <- superchick.mass[t-1] + sigfun(mean(chick.dee), pars$chick.mass.a, pars$chick.mass.mu, pars$chick.mass.lambda)
        
    if(puffin){ 
          
      punrat <- 1 - (((chick.mass[,t] / (pars$BM.chick.mortf * superchick.mass[t])) - pars$BM.chick.mortf) / (pars$BM.chick.leave - pars$BM.chick.mortf))
    }
    else{
      
      punrat <- time.chick.unattended[,t] / pars$Unattendance.hrs 
    }
    
    chick.mortality.from.unattendance[,t] <- rbinom(npairs, rep(1, npairs), vmax(0, vmin(1, punrat)))
    
    chick.mortality.from.lowmass[,t] <- In(chick.mass[,t] < (pars$BM.chick.mortf * superchick.mass[t])) 

    threshy <- mean(adult.mass[,,1]) * pars$BM.adult.abdn
    
    chick.mortality.from.abandonment[,t] <- In((adult.mass[,1,t] < threshy) | (adult.mass[,2,t] < threshy)) 

    abandoned[,t] <- 0 + (chick.mortality.from.unattendance[,t] | chick.mortality.from.lowmass[,t] | chick.mortality.from.abandonment[,t] | (abandoned[,t-1] == 1)) 
  }

  ## #################################

  ## out.energy <- list(adult.dee = adult.dee,
  ##                    independent.intake.rate = independent.intake.rate,
  ##                    final.intake.rate = final.intake.rate,
  ##                    energy.gained.per.hour = energy.gained.per.hour,
  ##                    total.energy.gained = total.energy.gained,
  ##                    energy.gap = energy.gap,
  ##                    energy.to.adult = energy.to.adult,
  ##                    energy.to.chick = energy.to.chick,
  ##                    energy.requirement = energy.requirement,
  ##                    total.energy.to.chick = total.energy.to.chick)
  
  ##out.times <- list(hours.for.requirement = hours.for.requirement,
  ##                  travel.hours.per.trip = travel.hours.per.trip,
  ##                  time.nest.minimum = time.nest.minimum,
  ##                  available.time = available.time, 
  ##                  flying.hours = flying.hours,
  ##                  hours.open.to.forage = hours.open.to.forage,
  ##                  foraging.hours = foraging.hours,
  ##                  surplus.hours = surplus.hours,
  ##                  time.nest = time.nest,
  ##                  time.rest = time.rest,
  ##                  nforagetrips = nforagetrips,
  ##                  time.chick.unattended = time.chick.unattended)
  
  out.final <- list(SPA = spatdat$SPA, 
                    adult.mass = adult.mass,
                    chick.mass = chick.mass,
                    abandoned = abandoned, 
                    chick.mortality.from.unattendance = chick.mortality.from.unattendance,
                    chick.mortality.from.lowmass = chick.mortality.from.lowmass,
                    chick.mortality.from.abandonment = chick.mortality.from.abandonment)

  ## list(energy = out.energy, times = out.times, outcomes = out.final)
  
  out.final
}

## ###############################################################################################

vsdm.summary <- function(out){

  d <- dim(out$outcomes$adult.mass)[3]
  
  new <- data.frame(SPA = out$outcomes$SPA,
                    final.adult.mass.f = out$outcomes$adult.mass[,1,d],
                    final.adult.mass.m = out$outcomes$adult.mass[,1,d],
                    chick.mortality.unattendance = out$outcomes$chick.mortality.from.unattendance[,d],
                    chick.mortality.lowmass = out$outcomes$chick.mortality.from.lowmass[,d],
                    chick.mortality.abandonment = out$outcomes$chick.mortality.from.abandonment[,d])

  new
}
              
## ###############################################################################################

vsdm.runmodel <- function(dat, species, meta, pars.full, pars.version=0, ppop=1, min.prey.adjust=1, seedval.location=NULL, seedval.behaviour=NULL, addWF=TRUE){

  which <- (pars.full$Version == pars.version) | is.na(pars.full$Version)

  if(any(pars.full$Version == pars.version & ! is.na(pars.full$Version))){
  
    param <- pars.full$Parameter[pars.full$Version == pars.version & ! is.na(pars.full$Version)]
    
    which <- which & ! ((pars.full$Parameter == param) & (is.na(pars.full$Version)))
  }

  pars.sub <- pars.full[which,]
  
  pars <- data.frame(t(pars.sub[,colnames(pars.sub) == species]))

  colnames(pars) <- pars.sub$Parameter

  if(pars$avoid.type == 2){ pars$avoid.type <- "large" }
  if(pars$avoid.type == 1){ pars$avoid.type <- "small" }
  if(pars$avoid.type == 0){ pars$avoid.type <- "old" }
    
  ## ###############
  
  colsizes <- tapply(meta$colonysize[,colnames(meta$colonysize) == species], meta$colonysize$SPA, sum)

  colsizes <- colsizes[match(levels(dat$Site), names(colsizes))]
  
  colsizes <- round(colsizes * ppop) 

  ## ###############
  
  if(addWF){
  
    print.noquote("Stage 1: Loading wind farm information....")
    
    dat <- addWF(dat, meta$wf, meta$colony)
  }

  nSPA <- length(levels(dat$Site))

  ## ###############

  if(is.null(seedval.location)){

    seedval.location <- round(runif(nSPA + 1, 0.5, 100000.5))
  }

  if(is.null(seedval.behaviour)){

    seedval.behaviour <- round(runif(1, 0.5, 100000.5))
  }

  ## ###############
    
  pars.null <- pars

  pars.null$disp.p <- pars.null$barrier.p <- 0

  ## ###############

  out <- as.list(NULL)

  ## ###############
  
  print.noquote("Stage 2: Simulating spatial locations....")

  print.noquote(">>>>  Baseline")  
  out$sim.baseline <- sim.batch.SPAs(dat, dat$NnG, npairs = colsizes, 
                                     wfstats = meta$wfstats[meta$wfstats$WindFarm == "NnG",], 
                                     seedval.location=seedval.location, pars = pars.null)
  
  print.noquote(">>>>  NnG")
  out$sim.NnG <- sim.batch.SPAs(dat, dat$NnG, npairs = colsizes, 
                                wfstats = meta$wfstats[meta$wfstats$WindFarm == "NnG",], 
                                seedval.location=seedval.location, pars = pars)
  
  print.noquote(">>>>  IC")
  out$sim.IC <- sim.batch.SPAs(dat, dat$IC, npairs = colsizes, 
                               wfstats = meta$wfstats[meta$wfstats$WindFarm == "IC",], 
                               seedval.location=seedval.location, pars = pars)
  
  print.noquote(">>>>  R3A")
  out$sim.R3A <- sim.batch.SPAs(dat, dat$R3A, npairs = colsizes, 
                                wfstats = meta$wfstats[meta$wfstats$WindFarm == "R3A",], 
                                seedval.location=seedval.location, pars = pars)
  print.noquote(">>>>  R3B")
  out$sim.R3B <- sim.batch.SPAs(dat, dat$R3B, npairs = colsizes, 
                                wfstats = meta$wfstats[meta$wfstats$WindFarm == "R3B",], 
                                seedval.location=seedval.location, pars = pars)
  
  ## ###############

  print.noquote("Stage 3: Simulating behaviour & demography....")

  total.prey <- rep(pars$total.prey, nSPA) 

  min.prey <- rep(pars$min.prey, nSPA) * min.prey.adjust 
  
  print.noquote(">>>>  Baseline")  
  out$res.baseline <- vsdm(out$sim.baseline, pars, total.prey = total.prey, puffin = (species == "Puffin"),
                           min.prey = min.prey, seedval.behaviour = seedval.behaviour, timestep = pars$timestep)

  print.noquote(">>>>  NnG")
  out$res.NnG <- vsdm(out$sim.NnG, pars, total.prey = total.prey, puffin = (species == "Puffin"),
                           min.prey = min.prey, seedval.behaviour = seedval.behaviour, timestep = pars$timestep)

  print.noquote(">>>>  IC")
  out$res.IC <- vsdm(out$sim.IC, pars, total.prey = total.prey, puffin = (species == "Puffin"),
                           min.prey = min.prey, seedval.behaviour = seedval.behaviour, timestep = pars$timestep)

  print.noquote(">>>>  R3A")
  out$res.R3A <- vsdm(out$sim.R3A, pars, total.prey = total.prey, puffin = (species == "Puffin"),
                           min.prey = min.prey, seedval.behaviour = seedval.behaviour, timestep = pars$timestep)

  print.noquote(">>>>  R3B")
  out$res.R3B <- vsdm(out$sim.R3B, pars, total.prey = total.prey, puffin = (species == "Puffin"),
                           min.prey = min.prey, seedval.behaviour = seedval.behaviour, timestep = pars$timestep)

  ## ###############

  out
}

## ###############################################################################################

vsdm.sensitivity <- function(dat, species, meta, pars.full, ppop=1, min.prey.adjust=1, seedval.location=NULL, seedval.behaviour=NULL){
  
  out <- as.list(NULL)

  nv <- max(pars.full$Version, na.rm=TRUE) + 1

  print.noquote("Stage 1: Loading wind farm information....")

  dat <- addWF(dat, meta$wf, meta$colony)

  ## ###############

  nSPA <- length(levels(dat$Site))

  if(is.null(seedval.location)){

    seedval.location <- round(runif(nSPA + 1, 0.5, 100000.5))
  }

  if(is.null(seedval.behaviour)){

    seedval.behaviour <- round(runif(1, 0.5, 100000.5))
  }
  
  ## ###############

  for(i in 1:(nv+1)){

    j <- i * (i <= nv)
    
    print.noquote("---------------------------------------")
    print.noquote(paste(">>>>>>>>>>>>> Version", i, date()))
    
    out[[i]] <- vsdm.runmodel(dat = dat, species = species, meta = meta,
                              pars.full = pars.full, pars.version = j,
                              ppop = ppop, min.prey.adjust = min.prey.adjust, addWF=FALSE,
                              seedval.location = seedval.location, seedval.behaviour = seedval.behaviour)
  }

  out
}

## ###############################################################################################

make.prey.map <- function(x, phi=100){

  fint <- function(sd, new){
    
    abs((qnorm(0.75, 0, sd) - qnorm(0.25, 0, sd)) - diff(quantile(new - median(new), c(0.25, 0.75))))
  }
  
  rescale <- function(x, phi){
    
    zot <- x - median(x)
    
    sd <- optimize(fint, interval=c(0,100), new=zot)$minimum
    
    zot <- rank(zot) / (length(zot) + 1)
    
    sdnew <- log(phi) / 3.92
    
    qnorm(zot, mean = 0, sd = sdnew)
  }
  
  qvals <- rescale(x, phi)
  
  fin <- exp(qvals)
  
  fin <- fin / sum(fin)
}

## ##############################################################################################

simlocs <- function(nn, bird.density, multip=10000000){

  ## Much quicker than rmultinom !!! (which is slow...)

  birdmul <- bird.density * multip
  
  birdmul <- round(birdmul)
  
  out <- rep(NA,multip)
  
  k1 <- 1
  
  for(i in 1:length(birdmul)){
    
    if(birdmul[i] > 0){
      
      k2 <- (k1 + birdmul[i] - 1)
      
      out[k1:k2] <- i
      
      k1 <- k2 + 1
    }
  }
  
  out <- out[! is.na(out)]
  
  id <- round(runif(nn, 0.5, length(out) + 0.5)) ## <<< STOCHASTIC >>>
  
  id <- out[id]

  id
}

## ##############################################################################################

sim.batch.SPAs <- function(dat, wf, npairs, wfstats, seedval.location, pars){
  
  sim.all.SPAs(dat, wf = wf, npairs = npairs, ntimes = pars$ntimes, disp.p = pars$disp.p, barrier.p = pars$barrier.p,
               wfstats = wfstats, avoid.type = pars$avoid.type, avoid.mean = pars$avoid.mean, avoid.sd = pars$avoid.sd, preyphi = pars$preyphi,
               seedval.location = seedval.location)
}


## ###############################################################################################

sim.one.SPA <- function(bird.density, zones, disp, npairs){
  
  nsites <- length(bird.density)
  
  ## ############
  
  loc.idx <- simlocs(npairs, bird.density) ## <<< STOCHASTIC >>>
  
  loc.zones <- zones[loc.idx]
  
  loc.moved <- (loc.zones == 4) & disp
  
  ## ############
  
  old.idx <- loc.idx 
  
  if(any(loc.moved)){
    
    s.moved <- sum(loc.moved)
    
    z35 <- (zones == 3 | zones == 5)
    
    mov.idx <- simlocs(s.moved, bird.density[z35] / sum(bird.density[z35])) ## <<< STOCHASTIC >>>
    
    loc.idx[loc.moved] <- ((1:nsites)[z35])[mov.idx]
  }
  
  loc.zones <- zones[loc.idx]
  
  ## ###########
  
  loc.moved <- factor(loc.moved)
  
  levels(loc.moved) <- c("", "0")
  
  loc.zones <- paste(loc.zones, loc.moved, sep="")
  
  loc.zones <- as.numeric(as.character(loc.zones))
  
  ## ###########

  data.frame(Index = loc.idx, OldIndex = old.idx, Zone = loc.zones) 
}

## ##############################################################################################

barrierdist <- function(dist, halfanglesweep, p){
  
  ## conservative formula
  ## distance based on bird reaching edge of colony, flying around perimeter, and then carrying on
  ## created 6 december 2013, last modified 8 december 2013
  
  angledir <- halfanglesweep * p
  
  opp  <- asin(sin(angledir) / sin(halfanglesweep))
  
  q <- pi - angledir - opp
  
  theta <- pi - 2 * (angledir + q)
  
  abs(dist * sin(halfanglesweep) * (theta - 2 * sin(theta/2)))
}

## ############################################################################

add.avoid <- function(sim, wfstats, barrier.p, avoid.type, avoid.mean, avoid.sd){
  
  if(avoid.type == "old"){
  
       tmp <- array(dim=dim(sim$Zone),
                    data = rnorm(prod(dim(sim$Zone)), avoid.mean, avoid.sd))  ## <<< STOCHASTIC >>>
  }
  else{
    
    if(avoid.type == "large"){
      
       lambda <- 0.5 * wfstats$large.anglerange
    }
    else{
      
        lambda <- 0.5 * wfstats$small.anglerange
    }
    
    lar <- lambda[match(sim$SPA, wfstats$SPA)] 
    
    lar <- array(dim=dim(sim$Zone), data=lar)
    
    dar <- wfstats$Dist[match(sim$SPA, wfstats$SPA)]
    
    dar <- array(dim=dim(sim$Zone), data=dar)
    
    mar <- wfstats$AngleMid[match(sim$SPA, wfstats$SPA)]
    
    mar <- array(dim=dim(sim$Zone), data=mar)
    
    pp <- abs((sim$Bearing - mar) / lar)
    
    pp[pp > 1] <- 1
      
    tmp <- barrierdist(dar, lar, pp)
  }
  
  bar <- array(dim=dim(sim$Zone),
               data = rbinom(prod(dim(sim$Zone)), 1, barrier.p))  ## <<< STOCHASTIC >>>
  
  nadd <- 2 * ((bar == 1) & (sim$Zone == 5 | sim$Zone == 6)) + ((bar == 1) * (sim$Zone == 50)) 
  
  sim$AvoidDist <- nadd * tmp

  sim
}

## ##############################################################################################

otherbirds <- function(sim){
  
  birds.in.square <- array(dim=dim(sim$GEBID))
  
  ntimes <- dim(sim$GEBID)[3]
  
  for(t in 1:ntimes){
    
    nos <- length(sim$GEBID[,1,t])
    
    a <- factor(c(sim$GEBID[,1,t], sim$GEBID[,2,t]))
    
    b <- as.numeric(summary(a, maxsum=100000))
    
    d <- a
    
    levels(d) <- b
    
    d <- as.numeric(as.character(d))
    
    birds.in.square[,1,t] <- d[1:nos]
    
    birds.in.square[,2,t] <- d[(nos+1):(2*nos)]
  }
  
  sim$Birds.in.square <- birds.in.square 

  sim
}

## ##############################################################################################

sim.all.SPAs <- function(dat, wf, npairs, ntimes, disp.p, barrier.p, wfstats, avoid.type, avoid.mean, avoid.sd, preyphi = 0, seedval.location){ 
    
  out <- as.list(NULL)

  nSPA <- length(levels(dat$Site))
  
  npairs.total <- sum(npairs)
  
  out$Depth <- out$Dist2Colony <- out$Prey.density <- out$Zone <- out$GEBID <- array(dim=c(npairs.total, 2, ntimes))

  out$Bearing <- out$DispDist <- array(dim=c(npairs.total, 2, ntimes))
  
  dat$GEBID <- apply(cogeb(dat$Longitude, dat$Latitude), 1, paste, collapse="-")

  out$SPA <- rep(NA, npairs.total)

  for(i in 1:nSPA){
    
    print.noquote(paste(levels(dat$Site)[i], date()))

    ## #######################

    set.seed(round(runif(1,1,10000)))
    
    disp <- rbinom(npairs[i], 1, disp.p) ## <<< STOCHASTIC >>> ## simulate *before* generating seed

    ## #######################

    set.seed(seedval.location[i])
    
    ## #######################

    which <- dat$Site == levels(dat$Site)[i]
    
    dat.i <- dat[which,]

    ## #######################

    if(preyphi == 0){
    
      dat.i$Prey.density <- 1/(dim(dat.i)[1])
    }
    else{

      dat.i$Prey.density <- make.prey.map(dat.i$Suitability4, preyphi)
    }
    
    bird.density <- exp(dat.i$Pred4) / sum(exp(dat.i$Pred4))

    ## #######################
    
    tmp <- sim.one.SPA(bird.density = bird.density, zones = wf[which], 
                   disp = disp, npairs = npairs[i] * ntimes * 2) ## <<< STOCHASTIC >>>
    
    dists <- geodist3(dat.i$Longitude[tmp$Index], dat.i$Latitude[tmp$Index], dat.i$Longitude[tmp$OldIndex], dat.i$Latitude[tmp$OldIndex])
    
    dists[is.nan(dists)] <- 0
    
    tmp$DispDist <- dists + dat.i$Dist2Colony[tmp$OldIndex] - dat.i$Dist2Colony[tmp$Index]
    
    ## table(tmp$Zone, abs(tmp$DispDist) > 0.001)
      
    ## ############################

    if(i == 1){

      prev <- 0
    }
    else{

      prev <- sum(npairs[1:(i-1)])
    }
    
    ix <- (1:npairs[i]) + prev
    
    ## ############################

    out$Depth[ix,,] <- dat.i$depth[tmp$Index]
    
    out$Bearing[ix,,] <- dat.i$Bearing[tmp$Index]
 
    out$Dist2Colony[ix,,] <- dat.i$Dist2Colony[tmp$Index]
    
    out$Prey.density[ix,,] <- dat.i$Prey.density[tmp$Index]
    
    out$Zone[ix,,] <- tmp$Zone

    out$DispDist[ix,,] <- tmp$DispDist
    
    out$SPA[ix] <- levels(dat$Site)[i]

    out$GEBID[ix,,] <- dat.i$GEBID[tmp$Index]
    
    ## ############################
  }

  set.seed(seedval.location[nSPA + 1])

  print.noquote(summary(factor(c(out$Zone))))

  out <- add.avoid(out, wfstats, barrier.p, avoid.type, avoid.mean, avoid.sd) ## <<< STOCHASTIC >>> 

  out <- otherbirds(out)
  
  out
}

## ##############################################################################################

getsummary.zones <- function(out){
  
  species <- names(out)
  
  windfarms <- gsub("sim.","",names(out[[1]])[1:5])

  new <- NULL

  nv <- length(out)
  
  for(i in 1:nv){
    
    SPAs <- unique(out[[i]][[1]]$SPA)

    nSPAs <- length(SPAs)

    for(j in 1:nSPAs){

      which <- (out[[i]][[j]]$SPA == SPAs[j])

      for(k in 1:5){
      
        smy <- data.frame(t(summary(factor(c(out[[i]][[k]]$Zone[which,,]), levels=c(1,3:6,30,50)))))

        nt <- dim(out[[i]][[k+5]]$adult.mass)[3]
            
        am <- out[[i]][[k+5]]$adult.mass[which,,]

        cm <- out[[i]][[k+5]]$chick.mass[which,]

        ab <- out[[i]][[k+5]]$abandoned[which,]
   
        zz <- c(out[[i]][[k]]$DispDist[which]) 

        if(any(zz != 0)){ 
        
          smy$DispDist <- mean(zz[zz != 0])
        }
        else{

          smy$DispDist <- NA
        }

        zz <- c(out[[i]][[k]]$AvoidDist[which])

        if(any(zz != 0)){ 
        
          smy$AvoidDist <- mean(zz[zz != 0])
        }
        else{

          smy$AvoidDist <- NA
        }
        
        smy$am.mean = mean(am[,,nt])
        smy$cm.mean = mean((cm[,nt])[ab[,nt] == 0]) 

        smy$am.sd = sd(am[,,nt])
        smy$cm.sd = sd((cm[,nt])[ab[,nt] == 0])
               
        smy$chickmortality = mean(ab[,nt])
        
        smy$Version <- i * (i < nv)

        smy$SPA <- SPAs[j]
        
        smy$WindFarm <- windfarms[k]
 
        new <- rbind(new, smy)
      }
    }
  }
  
  new <- new[,c(15:17,1:14)]

  colnames(new)[4:10] <- gsub("X","Z", colnames(new)[4:10])
 
  new[,4:10] <- t(apply(new[,4:10], 1, function(x){round(100 * x / sum(x), 4)})) 
  
  dd <- length(new$am.mean)
  
  new$am.change <- new$am.mean - rep(new$am.mean[seq(1,dd,5)], 1, each=5)

  new$cm.change <- new$cm.mean - rep(new$cm.mean[seq(1,dd,5)], 1, each=5)
  
  new <- new[,c(1:16,18:19,17)]

  new <- new[order(new$Version),]

  new <- new[order(new$SPA),]

  new
}

## ##############################################################################################

getlims <- function(WFzone, Bearing, Dist2Colony, eps=pi/100){

  ## ##########

  testpos <- (WFzone == 4 & Bearing > 0)
  
  if(any(testpos)){
  
    lims.hi <- range(Bearing[testpos])

    if(abs(pi - lims.hi[2]) < eps){ lims.hi[2] <- pi }
    
    if(abs(lims.hi[1]) < eps){ lims.hi[1] <- 0 }
  }
  else{

    lims.hi <- c(0,0)
  }

  ## ##########

  testneg <- (WFzone == 4 & Bearing < 0)
  
  if(any(testneg)){
    
    lims.lo <- range(Bearing[testneg])
   
    if(abs(-pi - lims.lo[1]) < eps){ lims.lo[1] <- -pi }
    
    if(abs(lims.lo[2]) < eps){ lims.lo[2] <- 0 }
  }
  else{

    lims.lo <- c(0,0)
  }

  ## ##########

  lims.dist <- mean(Dist2Colony[WFzone == 4])
  
  ## ##########

  c(lims.lo, lims.hi, lims.dist)
}
  
## ##############################################################################################

getzone <- function(WFzone, Bearing, Dist2Colony, lims){
    
  test.lims <- ((Bearing >= lims[1]) & (Bearing <= lims[2])) | ((Bearing >= lims[3]) & (Bearing <= lims[4]))
  
  test.dist <- (Dist2Colony > lims[5])
  
  test <- test.lims & test.dist
  
  ## ##########
  
  WFzone[test & (WFzone != 4) & (WFzone != 3)] <- 6
  
  WFzone[test & (WFzone == 3)] <- 5
  
  ## ##########
  
  WFzone
}

## ##############################################################################################

addWF <- function(dat, wf, colony, iom=TRUE, givelims=FALSE){
  
  lims <- as.list(NULL)

  dat$IC <- dat$NnG <- dat$R3AB <- dat$R3A <- dat$R3B <- dat$Bearing <- rep(1, dim(dat)[1])
  
  for(i in 1:length(levels(dat$Site))){

    which <- dat$Site == levels(dat$Site)[i]
    
    dat.i <- dat[which,]
    
    ni <- dim(dat.i)[1]

    ## ##########################

    long.ini <- (colony$long[as.character(colony$colony) == dat.i$Site[1]])[1]

    lat.ini <- (colony$lat[as.character(colony$colony) == dat.i$Site[1]])[1]

    dat.i$Bearing <- bearing(dat.i$Longitude, dat.i$Latitude, long.ini, lat.ini)
    
    ## ##########################
    
    zot <- cogeb(dat.i$Longitude, dat.i$Latitude) ; zot <- paste(zot[,1], zot[,2], sep="-")

    ## ##########################
    
    tmp <- cogeb(wf$Longitude, wf$Latitude) ; tmp <- paste(tmp[,1], tmp[,2], sep="-")
    
    ## ##########################
    
    dat.i$IC <- dat.i$NnG <- dat.i$R3AB <- dat.i$R3A <- dat.i$R3B <- rep(1, ni)
    
    ## ##########################
    
    dat.i$IC[unique(match(tmp[wf$WFcode == "IC-footprint"], zot))] <- 4
    dat.i$NnG[unique(match(tmp[wf$WFcode == "NnG-footprint"], zot))] <- 4
    dat.i$R3AB[unique(match(tmp[wf$WFcode == "R3AB-footprint"], zot))] <- 4
    dat.i$R3A[unique(match(tmp[wf$WFcode == "R3A-footprint"], zot))] <- 4
    dat.i$R3B[unique(match(tmp[wf$WFcode == "R3B-footprint"], zot))] <- 4
    
    ## ##########################
    
    dat.i$IC[unique(match(tmp[wf$WFcode == "IC-buffer"], zot))] <- 3
    dat.i$NnG[unique(match(tmp[wf$WFcode == "NnG-buffer"], zot))] <- 3
    dat.i$R3AB[unique(match(tmp[wf$WFcode == "R3AB-buffer"], zot))] <- 3
    dat.i$R3A[unique(match(tmp[wf$WFcode == "R3A-buffer"], zot))] <- 3
    dat.i$R3B[unique(match(tmp[wf$WFcode == "R3B-buffer"], zot))] <- 3

    ## ##########################

    dat$IC[which] <- dat.i$IC
    dat$NnG[which] <- dat.i$NnG
    dat$R3AB[which] <- dat.i$R3AB
    dat$R3A[which] <- dat.i$R3A
    dat$R3B[which] <- dat.i$R3B    
  }

  ## ##########################
  ## Has to be a new loop so that we can calculate lims first (needed for SPA areas without the WF):
  ## ##########################
  
  for(i in 1:length(levels(dat$Site))){

    which <- dat$Site == levels(dat$Site)[i]

    dat.i <- dat[which,]
    
    long.ini <- (colony$long[as.character(colony$colony) == dat.i$Site[1]])[1] 

    lat.ini <- (colony$lat[as.character(colony$colony) == dat.i$Site[1]])[1] 

    dat.i$Bearing <- bearing(dat.i$Longitude, dat.i$Latitude, long.ini, lat.ini)
    
    ##plot(dat.i$Longitude, dat.i$Latitude, col=rainbow(400)[1 + round((dat.i$Bearing + pi) * 180 / pi)])
    
    ##browser()
    
    ## ##########################
    
    ## dat$Bearing <- bearing(dat$Longitude, dat$Latitude, long.ini, lat.ini)
    dat.i$Dist2ThisColony <- geodist3(dat.i$Longitude, dat.i$Latitude, long.ini, lat.ini)
    
    lims.IC <- getlims(dat.i$IC, dat.i$Bearing, dat.i$Dist2ThisColony)
    lims.NnG <- getlims(dat.i$NnG, dat.i$Bearing, dat.i$Dist2ThisColony)
    lims.R3AB <- getlims(dat.i$R3AB, dat.i$Bearing, dat.i$Dist2ThisColony)
    lims.R3A <- getlims(dat.i$R3A, dat.i$Bearing, dat.i$Dist2ThisColony)
    lims.R3B <- getlims(dat.i$R3B, dat.i$Bearing, dat.i$Dist2ThisColony)

    if(givelims){

      lims[[i]] <- list(IC = lims.IC, NnG = lims.NnG,
                   R3AB = lims.R3AB, R3A = lims.R3A, R3B = lims.R3B)
    }
    
    ## ##########################
    
    dat.i$IC <- getzone(dat.i$IC, dat.i$Bearing, dat.i$Dist2Colony, lims.IC) 
    dat.i$NnG <- getzone(dat.i$NnG, dat.i$Bearing, dat.i$Dist2Colony, lims.NnG) 
    dat.i$R3AB <- getzone(dat.i$R3AB, dat.i$Bearing, dat.i$Dist2Colony, lims.R3AB) 
    dat.i$R3A <- getzone(dat.i$R3A, dat.i$Bearing, dat.i$Dist2Colony, lims.R3A) 
    dat.i$R3B <- getzone(dat.i$R3B, dat.i$Bearing, dat.i$Dist2Colony, lims.R3B) 
         
    ## ##########################
    
    dat$Bearing[which] <- dat.i$Bearing 
    
    dat$IC[which] <- dat.i$IC
    dat$NnG[which] <- dat.i$NnG
    dat$R3AB[which] <- dat.i$R3AB
    dat$R3A[which] <- dat.i$R3A
    dat$R3B[which] <- dat.i$R3B
  }

  if(givelims){

    lims
  }
  else{

    dat$Bearing <- toz(dat$Bearing)
    
    dat
  }
}

## ##############################################################################################

plotwf <- function(dat){
  
  for(i in 1:length(levels(dat$Site))){

    which <- dat$Site == levels(dat$Site)[i]

    dat.i <- dat[which,]

    par(mfrow=c(3,2), mar=c(1,1,1,1))
    
    cols <- c("darkblue", "darkblue", "orange", "red", "yellow", "purple")
    
    plot(dat.i$Longitude, dat.i$Latitude, col=cols[dat.i$IC], pch=20,
         cex=0.5, main=as.character(paste(dat.i$Site[1], "-IC", sep="")), axes=FALSE, xlab="", ylab="")
    
    plot(dat.i$Longitude, dat.i$Latitude, col=cols[dat.i$NnG], pch=20, cex=0.5, axes=FALSE, xlab="", ylab="", main="-NnG")
    plot(dat.i$Longitude, dat.i$Latitude, col=cols[dat.i$R3AB], pch=20, cex=0.5, axes=FALSE, xlab="", ylab="", main="-R3AB")
    plot(dat.i$Longitude, dat.i$Latitude, col=cols[dat.i$R3A], pch=20, cex=0.5, axes=FALSE, xlab="", ylab="", main="-R3A")
    plot(dat.i$Longitude, dat.i$Latitude, col=cols[dat.i$R3B], pch=20, cex=0.5, axes=FALSE, xlab="", ylab="", main="-R3B")

    browser()
  }
    
  NULL
}

## ##############################################################################################

